<?php
class SAPXMLClient {
    private $endpoint;
    private $username;
    private $password;

    public function __construct($endpoint, $username, $password) {
        $this->endpoint = $endpoint;
        $this->username = $username;
        $this->password = $password;
    }

    public function sendOrderXML($orderData) {
        // Generate XML
        $xml = $this->createOrderXML($orderData);
        
        // Send to SAP
        $response = $this->postXML($xml);
        
        // Parse response
        return $this->parseSAPResponse($response);
    }

    private function createOrderXML($data) {
        $xml = new SimpleXMLElement('<OrderRequest/>');
        
        // Header information
        $header = $xml->addChild('Header');
        $header->addChild('OrderID', $data['id']);
        $header->addChild('CustomerID', $data['customer_id']);
        $header->addChild('BranchCode', $data['branch_code']);
        $header->addChild('OrderDate', date('Y-m-d'));
        
        // Line items
        $items = $xml->addChild('Items');
        foreach ($data['items'] as $item) {
            $itemNode = $items->addChild('Item');
            $itemNode->addChild('Material', htmlspecialchars($item['product_sku']));
            $itemNode->addChild('Quantity', $item['quantity']);
            $itemNode->addChild('UOM', 'EA');
        }
        
        // Format XML
        $dom = dom_import_simplexml($xml)->ownerDocument;
        $dom->formatOutput = true;
        return $dom->saveXML();
    }

    private function postXML($xmlData) {
        $ch = curl_init($this->endpoint);
        
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/xml',
                'Authorization: Basic ' . base64_encode("$this->username:$this->password")
            ],
            CURLOPT_POSTFIELDS => $xmlData,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_TIMEOUT => 30
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        return [
            'body' => $response,
            'http_code' => $httpCode,
            'error' => $error,
            'request_xml' => $xmlData
        ];
    }

    private function parseSAPResponse($response) {
        if ($response['error']) {
            return [
                'success' => false,
                'error' => 'cURL Error: ' . $response['error'],
                'request' => $response['request_xml'],
                'response' => $response['body']
            ];
        }
        
        if ($response['http_code'] >= 400) {
            return [
                'success' => false,
                'error' => "HTTP Error: {$response['http_code']}",
                'request' => $response['request_xml'],
                'response' => $response['body']
            ];
        }
        
        try {
            $xml = simplexml_load_string($response['body']);
            if ($xml->getName() === 'ErrorResponse') {
                return [
                    'success' => false,
                    'error' => (string)$xml->ErrorMessage,
                    'code' => (string)$xml->ErrorCode,
                    'request' => $response['request_xml'],
                    'response' => $response['body']
                ];
            }
            
            return [
                'success' => true,
                'sap_order_id' => (string)$xml->OrderID,
                'confirmation' => (string)$xml->Confirmation,
                'request' => $response['request_xml'],
                'response' => $response['body']
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => 'XML Parse Error: ' . $e->getMessage(),
                'request' => $response['request_xml'],
                'response' => $response['body']
            ];
        }
    }
}