<?php
class SAPXMLClient {
    private $endpoint;
    private $username;
    private $password;

    public function __construct($endpoint, $username, $password) {
        $this->endpoint = $endpoint;
        $this->username = $username;
        $this->password = $password;
    }

    public function sendOrderXML($orderData) {
        // Generate XML
        $xml = $this->createOrderXML($orderData);
        
        // Send to SAP
        $response = $this->postXML($xml);
        
        // Parse response
        return $this->parseSAPResponse($response);
    }

    private function createOrderXML($data) {

        
        $xml = new SimpleXMLElement('<soapenv:Envelope xmlns:soapenv="http://schemas.xmlsoap.org/soap/envelope/" xmlns:urn="urn:sap-com:document:sap:rfc:functions"/>');    
            
        $header = $xml->addChild('<soapenv:Header');
        $header = $xml->addChild('<soapenv:Body');
        // Header information
        $header = $xml->addChild('<urn:ZTA_SAP_KFC_SO_INTEGRATION/>');
        $header->addChild('PORTAL_ID', $data['id']);
        $header->addChild('KUNNR', $data['branch_code']);
        $header->addChild('DATE', date('Y-m-d'));
        $header->addChild('PLANT', '1300');
        $header->addChild('SALES_ORG', '1000');
        $header->addChild('AUART', 'ZSIR');
        
        // Line items
        $items = $xml->addChild('ITEM_DETAIL');
        foreach ($data['items'] as $item) {
            $itemNode = $items->addChild('Item');
            $itemNode->addChild('MATERIAL', htmlspecialchars($item['product_sku']));
            $itemNode->addChild('TARGET_QTY', $item['quantity']);
            $itemNode->addChild('STORE_LOC', '1339');
        }
        
        // Format XML
        $dom = dom_import_simplexml($xml)->ownerDocument;
        $dom->formatOutput = true;
        return $dom->saveXML();
    }

    private function postXML($xmlData) {
        $ch = curl_init($this->endpoint);
        
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_HTTPHEADER => [
                'Content-Type: application/xml',
                'Authorization: Basic ' . base64_encode("$this->username:$this->password")
            ],
            CURLOPT_POSTFIELDS => $xmlData,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_SSL_VERIFYPEER => true,
            CURLOPT_TIMEOUT => 30
        ]);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        return [
            'body' => $response,
            'http_code' => $httpCode,
            'error' => $error,
            'request_xml' => $xmlData
        ];
    }

    private function parseSAPResponse($response) {
        if ($response['error']) {
            return [
                'success' => false,
                'error' => 'cURL Error: ' . $response['error'],
                'request' => $response['request_xml'],
                'response' => $response['body']
            ];
        }
        
        if ($response['http_code'] >= 400) {
            return [
                'success' => false,
                'error' => "HTTP Error: {$response['http_code']}",
                'request' => $response['request_xml'],
                'response' => $response['body']
            ];
        }
        
        try {
            $xml = simplexml_load_string($response['body']);
            if ($xml->getName() === 'ErrorResponse') {
                return [
                    'success' => false,
                    'error' => (string)$xml->ErrorMessage,
                    'code' => (string)$xml->ErrorCode,
                    'request' => $response['request_xml'],
                    'response' => $response['body']
                ];
            }
            
            return [
                'success' => true,
                'sap_order_id' => (string)$xml->OrderID,
                'confirmation' => (string)$xml->Confirmation,
                'request' => $response['request_xml'],
                'response' => $response['body']
            ];
        } catch (Exception $e) {
            return [
                'success' => false,
                'error' => 'XML Parse Error: ' . $e->getMessage(),
                'request' => $response['request_xml'],
                'response' => $response['body']
            ];
        }
    }
}