<?php
require_once '../../config.php';
include '../includes/auth.php';
include_once '../../includes/excel_export.php';

$current_date = date('Y-m-d');

// Initialize filter variables
$order_id = $_GET['order_id'] ?? '';
$start_date = $_GET['start_date'] ?? '';
$end_date = $_GET['end_date'] ?? '';
$status = $_GET['status'] ?? '';

// Build the base query
$query = "
   SELECT s.order_id, o.order_date, b.name AS branch_name, b.code AS branch_code,
   s.status, s.created_at, s.request_xml, s.response_xml 
   FROM sap_logs s 
   JOIN orders o ON o.id = s.order_id 
   JOIN branches b ON o.branch_id = b.id 
   WHERE 1=1
";

$params = [];
$types = '';

// Add default condition for initial load (last 3 days and future orders)
if (empty($start_date) && empty($end_date)) {
    $three_days_ago = date('Y-m-d', strtotime('-3 days'));
    $query .= " AND o.order_date >= ?";
    $params[] = $three_days_ago;
    $types .= 's';
}

// Add filters to the query if provided
if (!empty($order_id)) {
    $query .= " AND s.order_id = ?";
    $params[] = $order_id;
    $types .= 'i';
}

if (!empty($start_date)) {
    $query .= " AND o.order_date >= ?";
    $params[] = $start_date;
    $types .= 's';
}

if (!empty($end_date)) {
    $query .= " AND o.order_date <= ?";
    $params[] = $end_date . ' 23:59:59';
    $types .= 's';
}

if (!empty($status)) {
    $query .= " AND s.status = ?";
    $params[] = $status;
    $types .= 's';
}

$query .= " ORDER BY o.order_date, s.order_id DESC";

// Prepare and execute the query
if (!empty($params)) {
    $stmt = $conn->prepare($query);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $orders = $stmt->get_result();
} else {
    $orders = $conn->query($query);
}

// Get available statuses for filter dropdown
$statuses = ['pending', 'processed', 'failed', 'success'];

// Handle export request
if (isset($_GET['export']) && $_GET['export'] == 'excel') {
    $export_data = [];
    
    // Reset and execute the query again for export
    if (!empty($params)) {
        $stmt = $conn->prepare($query);
        $stmt->bind_param($types, ...$params);
        $stmt->execute();
        $orders_result = $stmt->get_result();
    } else {
        $orders_result = $conn->query($query);
    }
    
    while ($order = $orders_result->fetch_assoc()) {
        // Get order items for this order
        $stmt = $conn->prepare("
            SELECT oi.*, p.code AS product_code
            FROM order_items oi 
            JOIN products p ON oi.product_id = p.id 
            WHERE oi.order_id = ?
        ");
        $stmt->bind_param("i", $order['id']);
        $stmt->execute();
        $items_result = $stmt->get_result();
        
        while ($item = $items_result->fetch_assoc()) {
            $export_data[] = [
                'id' => $order['id'],
                'branch_code' => $order['branch_code'],
                'delivery_date' => $order['delivery_date'],
                'product_code' => $item['product_code'],
                'quantity' => $item['quantity']
            ];
        }
    }
    
    // Export to Excel
    exportOrdersToExcel($export_data, 'all_orders_' . date('Y-m-d'));
}
?>

<!DOCTYPE html>
<html>
<head>
    <title>View Orders</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="../../assets/css/style.css">
    <style>
        .filter-form {
            background-color: #f9f9f9;
            padding: 15px;
            border-radius: 5px;
            margin-bottom: 20px;
        }
        
        .filter-form .form-group {
            display: inline-block;
            margin-right: 15px;
            margin-bottom: 10px;
        }
        
        .filter-form label {
            display: block;
            margin-bottom: 5px;
        }
        
        .filter-actions {
            margin-top: 10px;
        }
        
        .clear-filters {
            margin-left: 10px;
        }
        
        .export-btn {
            background-color: #28a745;
            margin-left: 10px;
        }
        
        .export-btn:hover {
            background-color: #218838;
        }
    </style>
</head>
<body>
    <?php include '../includes/header.php'; ?>
    <div class="container">
        <h1>SAP Integration Log</h1>
        
        <div class="filter-form">
            <form method="get">
                <div class="form-group">
                    <label for="order_id">Order #:</label>
                    <input type="text" id="order_id" name="order_id" value="<?= htmlspecialchars($order_id) ?>">
                </div>
                
                <div class="form-group">
                    <label for="start_date">From Date:</label>
                    <input type="date" id="start_date" name="start_date" value="<?= htmlspecialchars($start_date) ?>">
                </div>
                
                <div class="form-group">
                    <label for="end_date">To Date:</label>
                    <input type="date" id="end_date" name="end_date" value="<?= htmlspecialchars($end_date) ?>">
                </div>
                
                <div class="form-group">
                    <label for="status">Status:</label>
                    <select id="status" name="status">
                        <option value="">All Statuses</option>
                        <?php foreach ($statuses as $s): ?>
                        <option value="<?= $s ?>" <?= $status == $s ? 'selected' : '' ?>><?= ucfirst($s) ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                
                <div class="filter-actions">
                    <button type="submit" class="btn">Apply Filters</button>
                    <a href="saplog.php" class="btn clear-filters">Clear Filters</a>
                </div>
            </form>
        </div>
        
        <table>
            <tr>
                <th>Order ID</th>
                <th>Branch</th>
                <th>Order Date</th>
                <th>Created Date</th>
                <th>Status</th>
                <th>Request XML</th>
                <th>Response XML</th>
            </tr>
            <?php while ($order = $orders->fetch_assoc()): ?>
            <tr>
                <td><?= $order['order_id'] ?></td>
                <td><?= htmlspecialchars($order['branch_name']) ?></td>
                <td><?= date('M j, Y', strtotime($order['order_date'])) ?></td>
                <td><?= date('M j, Y', strtotime($order['created_at'])) ?></td>
                <td>
                                <span class="badge bg-<?= $order['status'] == 'pending' ? 'failed' : 'success' ?>">
                                    <?= ucfirst($order['status']) ?>
                                </span>
                            </td>
                
                <td><?= htmlspecialchars($order['request_xml']) ?></td>
                <td><?= htmlspecialchars($order['response_xml']) ?></td>
                <td>
            
                </td>
            </tr>
            <?php endwhile; ?>
        </table>
    </div>
    <?php include '../includes/footer.php'; ?>
</body>
</html>